
#include <ESP8266WiFi.h>
#include <Wire.h>
#include <SimpleDHT.h>;
#include <EEPROM.h>
#include "ThingSpeak.h"

#define INTERVAL_SECONDS 600
//#define USE_DEEPSLEEP
#ifdef USE_DEEPSLEEP
#define NUM_MINMAX_TANK_LEVELS 1
#else
#define NUM_MINMAX_TANK_LEVELS 8
#endif

//Constants
char primary_WiFiSSID[] = "aaaa";
char primary_WiFiKey[] = "bbbb";
char secondary_WiFiSSID[] = "cccc";
char secondary_WiFiKey[] = "dddd";
unsigned long myChannelNumber = 123456;
const char * myWriteAPIKey = "0123456789ABCDEF";

#define DHTPIN 0     // what pin we're connected to
//#define DHTTYPE DHT22   // DHT 22  (AM2302)

#define BMP085_ADDRESS 0x77  // I2C address of BMP085


//Global objects and variables
WiFiClient client;
//DHT dht(DHTPIN, DHTTYPE); //// Initialize DHT sensor
SimpleDHT22 dht22;
const unsigned char OSS = 0;  // Oversampling Setting
// Calibration values
short ac1;
short ac2;
short ac3;
unsigned short ac4;
unsigned short ac5;
unsigned short ac6;
short b1;
short b2;
short mb;
short mc;
short md;

// b5 is calculated in bmp085GetTemperature(...), this variable is also used in bmp085GetPressure(...)
// so ...Temperature(...) must be called before ...Pressure(...).
long b5; 

// Read 1 byte from the BMP085 at 'address'
char bmp085Read(unsigned char address)
{
  unsigned char data;

  Wire.beginTransmission(BMP085_ADDRESS);
  Wire.write(address);
  Wire.endTransmission();

  Wire.requestFrom(BMP085_ADDRESS, 1);
  while(!Wire.available())
    ;

  return Wire.read();
}

// Read 2 bytes from the BMP085
// First byte will be from 'address'
// Second byte will be from 'address'+1
short bmp085ReadInt(unsigned char address)
{
  unsigned char msb, lsb;

  Wire.beginTransmission(BMP085_ADDRESS);
  Wire.write(address);
  Wire.endTransmission();

  Wire.requestFrom(BMP085_ADDRESS, 2);
  while(Wire.available()<2)
    ;
  msb = Wire.read();
  lsb = Wire.read();

  return (short) msb<<8 | lsb;
}

// Stores all of the bmp085's calibration values into global variables
// Calibration values are required to calculate temp and pressure
// This function should be called at the beginning of the program
void bmp085Calibration()
{
  ac1 = bmp085ReadInt(0xAA);
  ac2 = bmp085ReadInt(0xAC);
  ac3 = bmp085ReadInt(0xAE);
  ac4 = bmp085ReadInt(0xB0);
  ac5 = bmp085ReadInt(0xB2);
  ac6 = bmp085ReadInt(0xB4);
  b1 = bmp085ReadInt(0xB6);
  b2 = bmp085ReadInt(0xB8);
  mb = bmp085ReadInt(0xBA);
  mc = bmp085ReadInt(0xBC);
  md = bmp085ReadInt(0xBE);
}

// Calculate temperature in deg C
float bmp085GetTemperature(unsigned int ut){
  long x1, x2;

  x1 = (((long)ut - (long)ac6)*(long)ac5) >> 15;
  x2 = ((long)mc << 11)/(x1 + md);
  b5 = x1 + x2;

  float temp = ((b5 + 8)>>4);
  temp = temp /10;

  return temp;
}

// Calculate pressure given up
// calibration values must be known
// b5 is also required so bmp085GetTemperature(...) must be called first.
// Value returned will be pressure in units of Pa.
long bmp085GetPressure(unsigned long up){
  long x1, x2, x3, b3, b6, p;
  unsigned long b4, b7;

  b6 = b5 - 4000;
  // Calculate B3
  x1 = (b2 * (b6 * b6)>>12)>>11;
  x2 = (ac2 * b6)>>11;
  x3 = x1 + x2;
  b3 = (((((long)ac1)*4 + x3)<<OSS) + 2)>>2;

  // Calculate B4
  x1 = (ac3 * b6)>>13;
  x2 = (b1 * ((b6 * b6)>>12))>>16;
  x3 = ((x1 + x2) + 2)>>2;
  b4 = (ac4 * (unsigned long)(x3 + 32768))>>15;

  b7 = ((unsigned long)(up - b3) * (50000>>OSS));
  if (b7 < 0x80000000)
    p = (b7<<1)/b4;
  else
    p = (b7/b4)<<1;

  x1 = (p>>8) * (p>>8);
  x1 = (x1 * 3038)>>16;
  x2 = (-7357 * p)>>16;
  p += (x1 + x2 + 3791)>>4;

  long temp = p;
  return temp;
}

// Read the uncompensated temperature value
unsigned int bmp085ReadUT(){
  unsigned int ut;

  // Write 0x2E into Register 0xF4
  // This requests a temperature reading
  Wire.beginTransmission(BMP085_ADDRESS);
  Wire.write(0xF4);
  Wire.write(0x2E);
  Wire.endTransmission();

  // Wait at least 4.5ms
  delay(5);

  // Read two bytes from registers 0xF6 and 0xF7
  ut = bmp085ReadInt(0xF6);
  return ut;
}

// Read the uncompensated pressure value
unsigned long bmp085ReadUP(){

  unsigned char msb, lsb, xlsb;
  unsigned long up = 0;

  // Write 0x34+(OSS<<6) into register 0xF4
  // Request a pressure reading w/ oversampling setting
  Wire.beginTransmission(BMP085_ADDRESS);
  Wire.write(0xF4);
  Wire.write(0x34 + (OSS<<6));
  Wire.endTransmission();

  // Wait for conversion, delay time dependent on OSS
  delay(2 + (3<<OSS));

  // Read register 0xF6 (MSB), 0xF7 (LSB), and 0xF8 (XLSB)
  msb = bmp085Read(0xF6);
  lsb = bmp085Read(0xF7);
  xlsb = bmp085Read(0xF8);

  up = (((unsigned long) msb << 16) | ((unsigned long) lsb << 8) | (unsigned long) xlsb) >> (8-OSS);

  return up;
}

void writeRegister(int deviceAddress, byte address, byte val) {
  Wire.beginTransmission(deviceAddress); // start transmission to device 
  Wire.write(address);       // send register address
  Wire.write(val);         // send value to write
  Wire.endTransmission();     // end transmission
}

int readRegister(int deviceAddress, byte address){
  int v;
  Wire.beginTransmission(deviceAddress);
  Wire.write(address); // register to read
  Wire.endTransmission();

  Wire.requestFrom(deviceAddress, 1); // read a byte

  while(!Wire.available()) {
    // waiting
  }

  v = Wire.read();
  return v;
}

unsigned short min_tank_level, max_tank_level;
unsigned short last_tank_levels[NUM_MINMAX_TANK_LEVELS], cur_tank_level, total_tank_levels;

void setup() {
  int tries;

  // put your setup code here, to run once:
  pinMode(0, INPUT);
  pinMode(13, OUTPUT);
  digitalWrite(13, HIGH);
  pinMode(15, INPUT);
  pinMode(4, INPUT_PULLUP);
  pinMode(5, INPUT_PULLUP);
#ifdef USE_DEEPSLEEP
  pinMode(D0, WAKEUP_PULLUP);
#endif
  Serial.begin(115200);
  Serial.println("ESP8266 in normal mode");

  Wire.begin(4,5);
  bmp085Calibration();
//  dht.begin();

  // Connect to WiFi
  while( WiFi.status() != WL_CONNECTED ) {
    WiFi.begin(primary_WiFiSSID, primary_WiFiKey);
    tries = 20;
    while (WiFi.status() != WL_CONNECTED && tries--) {
      delay(500);
      Serial.print(".");
    }
    if( WiFi.status() == WL_CONNECTED )
      break;

    WiFi.begin(secondary_WiFiSSID, secondary_WiFiKey);
    tries = 20;
    while (WiFi.status() != WL_CONNECTED && tries--) {
      delay(500);
      Serial.print(".");
    }
  }
  Serial.println("");
  Serial.println("WiFi connected");
  
  // Print the IP address
  Serial.println(WiFi.localIP());
  ThingSpeak.begin(client);
  EEPROM.begin(4);
  EEPROM.get(0, min_tank_level);
  EEPROM.get(2, max_tank_level);
  if( min_tank_level == 65535 && max_tank_level == 65535 )
    max_tank_level = 0;
  if( max_tank_level < min_tank_level )
    max_tank_level = min_tank_level;
}

void loop() {
  float temperature, pressure, hum, temp;
  int i, val = 0;
  unsigned char reset_min_max_levels;

  Serial.println("Wake up");
  if( WiFi.status() != WL_CONNECTED ) {
    int tries = 20;
    Serial.println("Reconnect WiFi");
    WiFi.begin(primary_WiFiSSID, primary_WiFiKey);
    while (WiFi.status() != WL_CONNECTED && tries--)
      delay(500);
  
    if( WiFi.status() != WL_CONNECTED ) {
      tries = 20;
      Serial.println("Trying secondary");
      WiFi.begin(secondary_WiFiSSID, secondary_WiFiKey);
      while (WiFi.status() != WL_CONNECTED && tries--)
        delay(500);
    }

    if( WiFi.status() == WL_CONNECTED ) {
      Serial.println("Re-init ThingSpeak");
      ThingSpeak.begin(client);
    } else {
      Serial.println("Failed, sleeping");
      goto Sleep;
    }
  }
     
  Serial.println("Read bmp085");
  temperature = bmp085GetTemperature(bmp085ReadUT()); // in degrees Celcius
  pressure = bmp085GetPressure(bmp085ReadUP()) / 100.0; // in hPa
  Serial.println("DHT22");
  dht22.read2(DHTPIN, &temp, &hum, NULL);

/*
  Serial.print("temperature =");
  Serial.print(temperature);
  Serial.print(", pressure =");
  Serial.println(pressure);

  Serial.print("Humidity: ");
  Serial.print(hum);
  Serial.print("%, Temp: ");
  Serial.print(temp);
  Serial.println(" Celsius");
*/
  Serial.println("Read water level");
  digitalWrite(13, HIGH);
  delay(500);
  for( i = 0; i < 64; ++i ) {
    val += analogRead(0);
    if( i != 63 )
      delay(5);
  }
  digitalWrite(13, LOW);
  Serial.println("Done");
  last_tank_levels[cur_tank_level] = val;
  if( ++cur_tank_level == NUM_MINMAX_TANK_LEVELS )
    cur_tank_level = 0;
  if( total_tank_levels < NUM_MINMAX_TANK_LEVELS )
    ++total_tank_levels;
  reset_min_max_levels = digitalRead(15);
  if( reset_min_max_levels )
    Serial.print("resetting min/max levels");
  if( total_tank_levels == NUM_MINMAX_TANK_LEVELS || reset_min_max_levels ) {
    unsigned short min_level = 65535, max_level = 0, changed = 0;
    for( i = 0; i < NUM_MINMAX_TANK_LEVELS; ++i ) {
      if( last_tank_levels[i] < min_level )
        min_level = last_tank_levels[i];
      if( last_tank_levels[i] > max_level )
        max_level = last_tank_levels[i];
    }
    Serial.print("min = ");
    Serial.print(min_level);
    Serial.print(", max = ");
    Serial.println(max_level);
    if( reset_min_max_levels ) {
      min_tank_level = 65535;
      max_tank_level = 0;
    }
    if( max_level < min_tank_level && max_level > 5000 ) {
      min_tank_level = max_level;
      EEPROM.put(0, min_tank_level);
      changed = 1;
    }
    if( min_level > max_tank_level ) {
      max_tank_level = min_level;
      EEPROM.put(2, max_tank_level);
      changed = 1;
    }
    if( changed )
      EEPROM.commit();
  }

//  Serial.println(val);

  Serial.println("Uploading data...");
  if( min_tank_level >= max_tank_level ) {
    ThingSpeak.setField(1, 0.0f);
  } else {
    float level = ((val - min_tank_level) * 100.0 / (max_tank_level - min_tank_level));
    if( level < 0 )
      level = 0;
    else if( level > 100 )
      level = 100;
    ThingSpeak.setField(1, level);
  }
  ThingSpeak.setField(2, temp);
  ThingSpeak.setField(3, hum);
  ThingSpeak.setField(4, pressure);
  ThingSpeak.setField(5, val);
  ThingSpeak.setField(6, temperature);
  ThingSpeak.setField(7, min_tank_level);
  ThingSpeak.setField(8, max_tank_level);
  if( ThingSpeak.writeFields(myChannelNumber, myWriteAPIKey) == 200 )
    Serial.println("Done.");
  else
    Serial.println("Error.");

Sleep:
#ifdef USE_DEEPSLEEP
  ESP.deepSleep(INTERVAL_SECONDS * 1000000UL);
#else
  WiFi.forceSleepBegin();  //Putting the Modem off
  delay(INTERVAL_SECONDS*1000UL);
  WiFi.forceSleepWake();  //waking up the modem
#endif
}

